<?php

class SES {

	private const 			endpoint 	= [
		'production'	=> [
			'com'						=> 'https://hospedajes.ses.mir.es/hospedajes-web/ws/v1/comunicacion',
		],
		'testing'		=> [
			'com'						=> 'https://hospedajes.pre-ses.mir.es/hospedajes-web/ws/v1/comunicacion',
		]
	];

	private string $production			= 'testing';
	private bool $isProd				= false;

	private string			$tmpPath	= '';
	private string 			$certPath 	= '';

	protected $required					= [
		'general' => [
			'codigoEstablecimiento', 'codigoArrendador', 'tipoOperacion', 'tipoComunicacion'
		],
		'pago' => [],
		'alta' => [
			'referencia', 'fechaContrato', 'fechaEntrada', 'fechaSalida', 'pago', 'persona'
		],
		'persona' => [
			'rol', 'nombre', 'apellido1', 'fechaNacimiento', 'direccion'
		],
		'direccion' => [
			'direccion', 'codigoPostal', 'pais'
		]
	];

	protected array 		$login 		= [
		'username'							=> '',
		'password'							=> ''
	];

	private array 			$opts 		= [
		'codigoEstablecimiento'				=> ''
	];

	private array 			$header 	= [];

	protected ?DOMDocument 	$xml 		= null;

	function __construct ( string $username, string $password, array $opts = [] ) {

		$this -> checkRequiredKeys ( $opts, 'general' );

		$this -> tmpPath = realpath ( __DIR__ . '/_tmp' ) . DIRECTORY_SEPARATOR;
		$this -> certPath = realpath ( __DIR__ . '/_certs' ) . DIRECTORY_SEPARATOR;

		$this -> login ['username'] = $username;
		$this -> login ['password'] = $password;

		$this -> opts = array_merge ( $this -> opts, $opts );

		$this -> restartDom ();

	}

	public function setProduction ():void {
		$this -> production = 'production';
		$this -> isProd = true;
	}

	public function alta ( array $opts ):array {

		if ( !is_array ( $opts ) ) {
			self::catchError ( "Options should be an array in \"Alta Hospedajes\", found: " . gettype ( $opts ), 1004 );
		}

		$opts ['pago'] = array_merge ( $opts ['pago'], $this -> opts ['pago'] ?? [] );

		$this -> checkRequiredKeys ( $opts, 'alta' );

		$blocks = [
			'alta' => $this -> newElement ( 'alt:peticion', '', [
				'xmlns:alt' => 'http://www.neg.hospedajes.mir.es/altaParteHospedaje'
			] ),
			'solicitud' => $this -> newElement ( 'solicitud' ),
			'comunicacion' => $this -> newElement ( 'comunicacion' ),
			'contrato' => $this -> newElement ( 'contrato' ),
			'pago' => $this -> newElement ( 'pago' )
		];

		$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'referencia', $opts ['referencia'] ) );
		$this -> opts ['referencia'] = $opts ['referencia'] ?? '';
		$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'fechaContrato', $opts ['fechaContrato'] ?? date ( 'Y-m-d' ) ) );
		$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'fechaEntrada', $opts ['fechaEntrada'] ?? date ( 'Y-m-d\TH:i:s' ) ) );
		$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'fechaSalida', $opts ['fechaSalida'] ?? date ( 'Y-m-d\TH:i:s' ) ) );
		$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'numPersonas', $opts ['numPersonas'] ?? count ( $opts ['persona'] ?? [''] ) ) );
		if ( isset ( $opts ['numHabitaciones'] ) && !empty ( $opts ['numHabitaciones'] ) ) {
			$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'numHabitaciones', $opts ['numHabitaciones'] ) );
		}
		if ( isset ( $opts ['internet'] ) && !empty ( $opts ['internet'] ) ) {
			$blocks ['contrato'] -> appendChild ( $this -> newElement ( 'internet', $opts ['internet'] ) );
		}

		// Payments
		$blocks ['contrato'] -> appendChild ( $this -> pago ( $opts ['pago'] ) );
		// End Payments

		$blocks ['comunicacion'] -> appendChild ( $blocks ['contrato'] );
		// Persons
		foreach ( $opts ['persona'] as $v ) {
			$blocks ['comunicacion'] -> appendChild ( $this -> addPersona ( $v ) );
		}
		// End Persons
		$blocks ['solicitud'] -> appendChild ( $this -> newElement ( 'codigoEstablecimiento', $this -> opts ['codigoEstablecimiento'] ?? '' ) );
		$blocks ['solicitud'] -> appendChild ( $blocks ['comunicacion'] );
		$blocks ['alta'] -> appendChild ( $blocks ['solicitud'] );
		$this -> xml -> appendChild ( $blocks ['alta'] );

		$response = $this -> _prepare ();
		unset ( $this -> opts ['referencia'] );
		return $response;

	}

	public function addPersona ( array $opts ):?DOMElement {

		if ( !is_array ( $opts ) ) {
			self::catchError ( "Options should be an array in \"Persona\", found: " . gettype ( $opts ), 1004 );
		}

		$this -> checkRequiredKeys ( $opts, 'persona' );

		$persona = $this -> newElement ( 'persona' );
		$persona -> appendChild ( $this -> newElement ( 'rol', $opts ['rol'] ?? 'VI' ) );
		$persona -> appendChild ( $this -> newElement ( 'nombre', $opts ['nombre'] ?? '' ) );
		$persona -> appendChild ( $this -> newElement ( 'apellido1', $opts ['apellido1'] ?? '' ) );
		if ( isset ( $opts ['apellido2'] ) && !empty ( $opts ['apellido2'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'apellido2', $opts ['apellido2'] ) );
		}
		if ( isset ( $opts ['tipoDocumento'] ) && !empty ( $opts ['tipoDocumento'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'tipoDocumento', $opts ['tipoDocumento'] ) );
		}
		if ( isset ( $opts ['numeroDocumento'] ) && !empty ( $opts ['numeroDocumento'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'numeroDocumento', $opts ['numeroDocumento'] ) );
		}
		if ( isset ( $opts ['soporteDocumento'] ) && !empty ( $opts ['soporteDocumento'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'soporteDocumento', $opts ['soporteDocumento'] ) );
		}
		if ( isset ( $opts ['fechaNacimiento'] ) && !empty ( $opts ['fechaNacimiento'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'fechaNacimiento', $opts ['fechaNacimiento'] ) );
		}
		if ( isset ( $opts ['nacionalidad'] ) && !empty ( $opts ['nacionalidad'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'nacionalidad', $opts ['nacionalidad'] ) );
		}
		if ( isset ( $opts ['sexo'] ) && !empty ( $opts ['sexo'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'sexo', $opts ['sexo'] ) );
		}
		// Adress
		$persona -> appendChild ( $this -> direccion ( $opts ['direccion'] ) );
		if ( isset ( $opts ['telefono'] ) && !empty ( $opts ['telefono'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'telefono', $opts ['telefono'] ) );
		}
		if ( isset ( $opts ['telefono2'] ) && !empty ( $opts ['telefono2'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'telefono2', $opts ['telefono2'] ) );
		}
		if ( isset ( $opts ['correo'] ) && !empty ( $opts ['correo'] ) ) {
			$persona -> appendChild ( $this -> newElement ( 'correo', $opts ['correo'] ) );
		}
		if ( isset ( $opts ['parentesco'] ) && !empty ( $opts ['parentesco'] ) && $opts ['parentesco'] != 'NULL' ) {
			$persona -> appendChild ( $this -> newElement ( 'parentesco', $opts ['parentesco'] ) );
		}

		return $persona;

	}

	public function addOpt ( string $key, string|int|array|object|null $value ):void {

		$this -> opts [$key] = $value;

	}

	public function addHeader ( string $key, string $value ):void {

		$this -> header [$key] = $value;

	}

	private function pago ( array $opts = [] ):DOMElement {

		$this -> checkRequiredKeys ( $opts, 'pago' );

		$this -> opts ['pago'] = array_merge ( $this -> opts ['pago'] ?? [], $opts );

		$pago = $this -> newElement ( 'pago' );
		$pago -> appendChild ( $this -> newElement ( 'tipoPago', $opts ['tipoPago'] ?? 'OTRO' ) );
		if ( isset ( $opts ['fechaPago'] ) && !empty ( $opts ['fechaPago'] ) ) {
			$pago -> appendChild ( $this -> newElement ( 'fechaPago', $opts ['fechaPago'] ) );
		}
		if ( isset ( $opts ['medioPago'] ) && !empty ( $opts ['medioPago'] ) ) {
			$pago -> appendChild ( $this -> newElement ( 'medioPago', $opts ['medioPago'] ) );
		}
		if ( isset ( $opts ['titular'] ) && !empty ( $opts ['titular'] ) ) {
			$pago -> appendChild ( $this -> newElement ( 'titular', $opts ['titular'] ) );
		}
		if ( isset ( $opts ['caducidadTarjeta'] ) && !empty ( $opts ['caducidadTarjeta'] ) ) {
			$pago -> appendChild ( $this -> newElement ( 'caducidadTarjeta', $opts ['caducidadTarjeta'] ) );
		}
		return $pago;

	}

	private function direccion ( array $opts ):DOMElement {

		$this -> checkRequiredKeys ( $opts, 'direccion' );

		$adress = $this -> newElement ( 'direccion' );

		$adress -> appendChild ( $this -> newElement ( 'direccion', $opts ['direccion'] ?? '' ) );
		if ( isset ( $opts ['direccionComplementaria'] ) && !empty ( $opts ['direccionComplementaria'] ) ) {
			$adress -> appendChild ( $this -> newElement ( 'direccionComplementaria', $opts ['direccionComplementaria'] ) );
		}
		if ( $opts ['pais'] === 'ESP' ) {
			$adress -> appendChild ( $this -> newElement ( 'codigoMunicipio', $opts ['codigoMunicipio'] ?? '' ) );
		} else {
			if ( isset ( $opts ['codigoMunicipio'] ) && !empty ( $opts ['codigoMunicipio'] ) ) {
				$adress -> appendChild ( $this -> newElement ( 'codigoMunicipio', $opts ['codigoMunicipio'] ) );
			}
		}
		if ( $opts ['pais'] !== 'ESP' ) {
			$adress -> appendChild ( $this -> newElement ( 'nombreMunicipio', $opts ['nombreMunicipio'] ?? '' ) );
		} else {
			if ( isset ( $opts ['nombreMunicipio'] ) && !empty ( $opts ['nombreMunicipio'] ) ) {
				$adress -> appendChild ( $this -> newElement ( 'nombreMunicipio', $opts ['nombreMunicipio'] ) );
			}
		}
		$adress -> appendChild ( $this -> newElement ( 'codigoPostal', $opts ['codigoPostal'] ?? '' ) );
		$adress -> appendChild ( $this -> newElement ( 'pais', $opts ['pais'] ?? 'ESP' ) );

		return $adress;

	}

	private function restartDom ():void {

		$this -> xml = new DOMDocument ( '1.0', 'UTF-8' );
		$this -> xml -> formatOutput = true;

	}

	private function newElement ( string $element, string $value = '', array $attributes = [] ):DOMElement {

		$temp = $this -> xml -> createElement ( $element, $this -> sanitizeInput ( $value ) );

		foreach ( $attributes as $k => $v ) {
			$temp -> setAttribute ( $k, $v );
		}

		return $temp;

	}

	private function _minize ( string $xml ):string {

		return trim (
			preg_replace ( [
				'/(\n|^)(\x20+|\t)/',
				'/(\n|^)\/\/(.*?)(\n|$)/',
				'/\n/',
				'/\r/',
				'/\<\!--.*?-->/',
				'/(\x20+|\t)/',
				'/\>\s+\</',
				'/(\"|\')\s+\>/',
				'/=\s+(\"|\')/',
				'/\.(jpg|png|gif|webp|php|mp4)/im',
				'/\.#(jpg|png|gif|webp|php|mp4)/'
			],[
				"\n",
				"\n",
				" ",
				" ",
				"",
				" ",
				"><",
				"$1>",
				"=$1",
				"",
				".$1"
			], $xml )
		);

	}

	private function sanitizeInput ( string $input ):string {
		return htmlspecialchars ( trim ( $input ) );
	}

	private function checkRequiredKeys ( array $array, string $required ):void {
		if ( !isset ( $this -> required [$required] ) ) {
			self::catchError ( "The required values: {$required}, doesn't exists", 1002 );
		}
		$missing = array_diff ( $this -> required [$required], array_keys ( $array ) );
		if ( !empty ( $missing ) ) {
			self::catchError ( "Missing next values: " . implode ( ', ', $missing ), 1003 );
		}
	}

	private function zip ( string $xml ):bool|string {

		$tmpZip = $this -> tmpPath . 'SES_Hospedaje.zip';

		$zip = new ZipArchive ();

		if ( $zip -> open ( $tmpZip, ZipArchive::CREATE | ZipArchive::OVERWRITE ) === TRUE ) {
			$zip -> addFile ( $xml, basename ( $xml ) );
			$zip -> close ();

			$base64 = file_get_contents ( $tmpZip );
			$base64 = base64_encode ( $base64 );

			unlink ( $tmpZip );

			return $base64;
		}

		return false;

	}

	protected function envelope ( string $xml ):DOMElement {

		$blocks = [
			'envelope' => $this -> newElement ( 'soapenv:Envelope', '', [
				'xmlns:soapenv'					=> 					'http://schemas.xmlsoap.org/soap/envelope/',
				'xmlns:com'						=>					'http://www.soap.servicios.hospedajes.mir.es/comunicacion'
			] ),
			'body' => $this -> newElement ( 'soapenv:Body' ),
			'comRequest' => $this -> newElement ( 'com:comunicacionRequest' ),
			'petition' => $this -> newElement ( 'peticion' ),
			'header' => $this -> newElement ( 'cabecera' ),
			'request' => $this -> newElement ( 'solicitud', $xml )
		];

		$blocks ['envelope'] -> appendChild ( $this -> newElement ( 'soapenv:Header' ) );

		$blocks ['header'] -> appendChild ( $this -> newElement ( 'codigoArrendador', $this -> opts ['codigoArrendador'] ) );
		$blocks ['header'] -> appendChild ( $this -> newElement ( 'aplicacion', $this -> opts ['aplicacion'] ?? 'SES Hospedajes' ) );
		$blocks ['header'] -> appendChild ( $this -> newElement ( 'tipoOperacion', $this -> opts ['tipoOperacion'] ) );
		$blocks ['header'] -> appendChild ( $this -> newElement ( 'tipoComunicacion', $this -> opts ['tipoComunicacion'] ) );

		$blocks ['petition'] 	-> appendChild ( $blocks ['header'] );
		$blocks ['petition'] 	-> appendChild ( $blocks ['request'] );
		$blocks ['comRequest'] 	-> appendChild ( $blocks ['petition'] );
		$blocks ['body'] 		-> appendChild ( $blocks ['comRequest'] );
		$blocks ['envelope'] 	-> appendChild ( $blocks ['body'] );

		return $blocks ['envelope'];

	}

	public function getCodes ( ?string $type = null ):array {

		$type = $type ? strtoupper ( $type ) : $type;

		$blocks = [
			'envelope' => $this -> newElement ( 'soapenv:Envelope', '', [
				'xmlns:soapenv'					=> 'http://schemas.xmlsoap.org/soap/envelope/',
				'xmlns:com'						=> 'http://www.soap.servicios.hospedajes.mir.es/comunicacion'
			] ),
			'body' => $this -> newElement ( 'soapenv:Body' ),
			'request' => $this -> newElement ( 'com:catalogoRequest' ),
			'petition' => $this -> newElement ( 'peticion' )
		];

		if ( $type && !in_array ( $type, [
			'TIPO_DOCUMENTO',
			'SEXO',
			'TIPO_PAGO',
			'TIPO_PARENTESCO',
			'TIPO_ESTABLECIMIENTO',
			'TIPO_VEHICULO',
			'TIPO_MARCA_VEHICULO',
			'TIPO_COLOR',
			'TIPO_PERMISO_CONDUCIR'
		] ) ) {
			self::catchError ( "Type check \"{$type}\" doesn't exists", 1006 );
		}

		$blocks ['petition'] -> appendChild ( $this -> newElement ( 'catalogo', $type ?? '' ) );

		$blocks ['request'] -> appendChild ( $blocks ['petition'] );
		$blocks ['body'] -> appendChild ( $blocks ['request'] );
		$blocks ['envelope'] -> appendChild ( $this -> newElement ( 'soapenv:Header' ) );
		$blocks ['envelope'] -> appendChild ( $blocks ['body'] );

		$xml = new DOMDocument ( '1.0', 'UTF-8' );
		$xml -> appendChild ( $xml -> importNode ( $blocks ['envelope'], true ) );
		$xml = $xml -> saveXML ();

		$authToken = base64_encode ( "{$this -> login ['username']}:{$this -> login ['password']}" );

		$this -> header = [];

		$this -> addHeader ( "Content-Type", "text/xml; charset=utf-8" );
		$this -> addHeader ( "SOAPAction", "\"comunicacionRequest\"" );
		$this -> addHeader ( "Authorization", "Basic {$authToken}" );
		$this -> addHeader ( "Accept", "application/xml" );

		return $this -> _send (
			$this -> _minize ( $xml )
		);

	}

	private function _prepare ():bool|array {

		if ( !file_exists ( realpath ( $this -> certPath . 'cert.pem' ) ) || !file_exists ( realpath ( $this -> certPath . 'cert_private.pem' ) ) ) {
			self::catchError ( "Can't load certificates. Please, check if certificates \"cert.pem\" and \"cert_private.pem\" are located at: {$this -> certPath}", 1001 );
		}

		$xml = $this -> xml -> saveXML ();
		$xml = $this -> _minize ( $xml );

		$tmpFile = "ses_hospedaje_{$this -> opts ['referencia']}_" . time () . ".xml";
		file_put_contents ( $this -> tmpPath . $tmpFile, $xml );

		$this -> restartDom ();
		$this -> xml -> appendChild ( $this -> envelope ( $this -> zip ( $this -> tmpPath . $tmpFile ) ) );
		unlink ( $this -> tmpPath . $tmpFile );

		$xml = $this -> xml -> saveXML ();
		$xml = $this -> _minize ( $xml );

		if ( !$this -> isProd ) {
			echo $xml;
			return true;
		}

		$authToken = base64_encode ( "{$this -> login ['username']}:{$this -> login ['password']}" );

		$this -> addHeader ( "Content-Type", "text/xml; charset=utf-8" );
		$this -> addHeader ( "SOAPAction", "\"comunicacionRequest\"" );
		$this -> addHeader ( "Authorization", "Basic {$authToken}" );
		$this -> addHeader ( "Accept", "application/xml" );

		return $this -> _send (
			$this -> _minize ( $xml )
		);

	}

	private function _send ( string $xml, string $endpoint = 'com' ):array {

		$ch = curl_init ();

		if ( !$this -> isProd ) {ob_start ();}

		curl_setopt_array ( $ch, [
			CURLOPT_URL             => self::endpoint [$this -> production] [$endpoint],
			CURLOPT_VERBOSE         => 1,
			CURLOPT_SSL_VERIFYPEER  => 0,
			CURLOPT_SSL_VERIFYHOST  => 2,
			CURLOPT_SSLKEY          => $this -> certPath . 'cert_private.pem',
			CURLOPT_SSLCERT         => $this -> certPath . 'cert.pem',
			CURLOPT_SSLCERTTYPE     => "PEM",
			// CURLOPT_KEYPASSWD		=> '',
			CURLOPT_CERTINFO		=> !$this -> isProd,
			CURLOPT_RETURNTRANSFER  => 1,
			CURLOPT_POST            => 1,
			CURLOPT_HTTPHEADER      => array_map ( fn ($k) => "{$k}: {$this -> header [$k]}", array_keys ( $this -> header ) ),
			CURLOPT_POSTFIELDS      => $xml
		] );

		$response   = curl_exec ( $ch );
		if ( !$this -> isProd ) {$verbose = ob_get_clean ();}

		$httpCode   = curl_getinfo ( $ch, CURLINFO_HTTP_CODE );
		$error      = curl_error ( $ch );
		$errorNr    = curl_errno ( $ch );
		if ( !$this -> isProd ) {$certInfo = curl_getinfo ( $ch, CURLINFO_CERTINFO );}

		curl_close ( $ch );

		if ( !$this -> isProd ) {

			file_put_contents ( $this -> tmpPath . 'debug_request.xml', $xml );
			file_put_contents ( $this -> tmpPath . 'debug_response.xml', $response );
			// file_put_contents ( $this -> tmpPath . 'debug_verbose.log', $verbose );
			file_put_contents ( $this -> tmpPath . 'debug_info.json', json_encode (
				[
					'authorization'			=> base64_encode ( "{$this -> login ['username']}:{$this -> login ['password']}" ),
					'http_code'				=> $httpCode,
					'error'					=> $error,
					'errno'					=> $errorNr,
					'certinfo'				=> $certInfo
				],
				JSON_PRETTY_PRINT
			) );

		}

		switch ( $httpCode ) {
			case 200:
				return $this -> processResponse ( $response );
				break;
			case 404:
				self::catchError ( "URL not found: \"" . self::endpoint [$this -> production] [$endpoint] . "\"", 404 );
				break;
			case 401:
				selF::catchError ( "Username or Password are not valid!", 401 );
				break;
			default:
				self::catchError ( "({$httpCode}): {$error}", $errorNr );
				break;
		}

		return [];

	}

	private function processResponse ( string $resp ):array {

		libxml_use_internal_errors ( true );

		$xml = simplexml_load_string ( $resp );
		if ( $xml === false ) {
			self::catchError ( "Invalid XML response: " . libxml_get_errors (), 1100 );
		}

		$namespaces = $xml -> getNamespaces ( true );
		if ( !isset ( $namespaces ['SOAP-ENV'] ) ) {
			self::catchError ( "SOAP namespace SOAP-ENV not found!", 1102 );
		}

		$body = $xml -> children ( $namespaces ['SOAP-ENV'] ) -> Body;
		if ( !$body ) {
			self::catchError ( "No SOAP body found!", 1101 );
		}

		$com = $body -> children ( $namespaces ['ns3'] ) ?? null;
		if ( !$com ) {
			self::catchError ( "ns3 Comunication not found!", 1103 );
		}

		$response = [
			'type' =>  $com -> getName ()
		];

		switch ( $response ['type'] ) {
			case 'catalogoResponse':

				$val = $com -> children () -> resultado ?? null;
				if ( !$val ) {
					self::catchError ( "No result node found!", 1105 );
				}

				switch ( (int) $val -> codigo ) {
					case 0:
						$val = $com -> children () -> respuesta ?? null;
						if ( !$val ) {
							self::catchError ( "No response node found!", 1104 );
						}

						$response ['code'] = (int) $val -> codigo;
						$response ['values'] = [];

						foreach ( $val -> resultado -> tupla as $v ) {

							$response ['values'] [] = [
								'value' => (string) $v -> codigo,
								'description' => (string) $v -> descripcion
							];

						}

						break;
					default:
						break;
				}

				break;

			case 'comunicacionResponse':

				$val = $com -> children () -> respuesta ?? null;
				if ( !$val ) {
					self::catchError ( "No response node found!", 1104 );
				}

				switch ( (int) $val -> codigo ) {
					case 0:
						$response ['code'] = (int) $val -> codigo;
						$response ['lote'] = (string) $val -> lote;
						break;
					default:
						$response ['code'] = (int) $val -> codigo;
						$response ['msg'] = (string) $val -> descripcion;
						break;
				}

				break;

			default:

				$response = [
					'type' => 'unknown',
					'msg' => 'Unknown response'
				];

				break;
		}

		return $response;

	}

	private function catchError ( string $message, ?int $code = null ):void {

		try {
			throw new Exception ( $message, $code );
		} catch ( Exception $e ) {
			echo "Error [{$e -> getCode ()}]: {$e -> getMessage ()}";
			exit ();
		}

	}

	public function genCerts ( string $p12, string $password, string $type = "PEM", string $openSSLPath = '/usr/bin/openssl' ):bool {

		if ( !file_exists ( $p12 ) ) {
			self::catchError ( "Can't found cert in \"{$p12}\"", 1006 );
		}

		if ( !is_executable ( $openSSLPath ) ) {
			self::catchError ( "OpenSSL not found at path {$openSSLPath}. Please check your System Variables and include openssl.", 1010 );
		}
        
        $openSSLPath = escapeshellarg ( $openSSLPath );
		$p12 = escapeshellarg ( $p12 );
		$pfx = escapeshellarg ( $this -> certPath . 'cert.pfx' );
		$cert = escapeshellarg ( $this -> certPath . 'cert.pem' );
		$private = escapeshellarg ( $this -> certPath . 'cert_private.pem' );
		$password = escapeshellarg ( "pass:{$password}" );

		$shell = "{$openSSLPath} pkcs12 -in {$p12} -clcerts -nokeys -out {$cert} -passin {$password}";
		exec ( $shell, $out1, $code1 );
		$shell = "{$openSSLPath} pkcs12 -in {$p12} -nocerts -nodes -out {$private} -passin {$password}";
		exec ( $shell, $out2, $code2 );

		if (
			$code1 === 0 && $code2 === 0 &&
			file_exists ( $this -> certPath . 'cert.pem' ) &&
			file_exists ( $this -> certPath . 'cert_private.pem' )
		) {
			switch ( $type ) {
				case "P12":
				case "PFX":
					$shell = "{$openSSLPath} pkcs12 -export -out {$pfx} -inkey {$private} -in {$cert} -passout pass:";
					exec ( $shell, $out3, $code3 );
					if (
						$code3 === 0 &&
						file_exists ( $this -> certPath . 'cert.pfx' )
					) {
						unlink ( $this -> certPath . 'cert.pem' );
						unlink ( $this -> certPath . 'cert_private.pem' );
					}
					break;
			}

			return true;

		}

		return false;

	}

}

?>